﻿#******************************************************************************
#
#    ＊ ポーカー
#
#  --------------------------------------------------------------------------
#    バージョン ： 1.0.2
#    対      応 ： RPGツクールVX Ace : RGSS3
#    制  作  者 ： ＣＡＣＡＯ
#    配  布  元 ： http://cacaosoft.webcrow.jp/
#  --------------------------------------------------------------------------
#   == 概    要 ==
#
#    ： ポーカーの機能を追加します。
#
#  --------------------------------------------------------------------------
#   == 注意事項 ==
#
#    ※ 動作には、トランプの画像が必要です。
#
#  --------------------------------------------------------------------------
#   == 使用方法 ==
#
#    ★ ポーカーの開始
#     イベントコマンド「スクリプト」に start_poker と記述
#
#
#******************************************************************************


#==============================================================================
# ◆ 設定項目
#==============================================================================
module CAO
module Poker
  #--------------------------------------------------------------------------
  # ◇ コインを保存するＥＶ変数の番号
  #--------------------------------------------------------------------------
  IDV_COIN = 7
  #--------------------------------------------------------------------------
  # ◇ 掛け金で最初に表示される枚数
  #--------------------------------------------------------------------------
  DEFAULT_COIN = 100
  #--------------------------------------------------------------------------
  # ◇ 掛け金の増減数
  #--------------------------------------------------------------------------
  BET_RATE = 10
  #--------------------------------------------------------------------------
  # ◇ 掛け金の最大枚数
  #--------------------------------------------------------------------------
  MAX_BET = 100
  #--------------------------------------------------------------------------
  # ◇ 掛け金の増加量 ( n 倍)
  #--------------------------------------------------------------------------
  DIVIDEND_COVER = [0, 1, 2, 3, 5, 7, 10, 50, 100, 500]
  #--------------------------------------------------------------------------
  # ◇ 配当金の下三桁が 000 の場合 K と表示する
  #--------------------------------------------------------------------------
  ADDR_DIVIDEND = true
  #--------------------------------------------------------------------------
  # ◇ 役の名前
  #--------------------------------------------------------------------------
  VOCAB_HANDS = [ "High card", "One pair", "Two pair", "Three of a kind",
                  "Straight", "Flush", "Full House", "Four of a kind",
                  "Straight Flush", "Royal Flush"               ]
  #--------------------------------------------------------------------------
  # ◇ 掛け金の名称と単位
  #--------------------------------------------------------------------------
  VOCAB_COIN = "Balance"
  VOCAB_COIN_U = " coins"
  #--------------------------------------------------------------------------
  # ◇ 掛け金選択時のメッセージ
  #--------------------------------------------------------------------------
  VOCAB_BET  = "Please set your wager"
  #--------------------------------------------------------------------------
  # ◇ １ゲーム終了後のメッセージ
  #--------------------------------------------------------------------------
  VOCAB_WIN  = "Congratulations.\n You won \\C[14]%s\\C[0] coins."
  VOCAB_LOSE = "That's too bad.\nYou lost and got \\C[14]%s\\C[0] coins."
  #--------------------------------------------------------------------------
  # ◇ ヘルプウィンドウの文章
  #--------------------------------------------------------------------------
  VOCAB_HELP = "Ａ : Game description"
  #--------------------------------------------------------------------------
  # ◇ ゲームの説明
  #--------------------------------------------------------------------------
  MANUAL_LINE = 11            # 行数
  TEXT_MANUAL = <<-'[EOS]'    # 文章 (次の行から [EOS] まで)
  
\C[16]＜Setting the amount of your bid＞\C[0]
 ・\C[2]↑/↓ key\C[0] Increase/decrease your bid
 ・\C[2]Ｃ key\C[0] Confirm your bid and start the game
 ・\C[2]Ｂ key\C[0] Cancel

\C[16]＜Card selection＞\C[0]
 ・\C[2]←/→ key\C[0] Change the highlighted card
 ・\C[2]Ｃ key\C[0] Flip the selected card
 ・\C[2]Ｂ key\C[0] Replace face-down cards and confirm your hand
  [EOS]
  #--------------------------------------------------------------------------
  # ◇ 効果音
  #--------------------------------------------------------------------------
  SOUND_START   = "Flash2"      # カードを表にして選択可能になったとき
  SOUND_WIN     = "Applause1"   # 役あり
  SOUND_LOSE    = "Scream"      # 役なし
  SOUND_BET     = "Shop"        # 掛け金を決定したとき
  SOUND_COIN    = "Coin"        # 掛け金を変更したとき
  SOUND_DEAL    = "Book2"       # 手札を配るとき
  SOUND_SLOUGH  = "Book2"       # 手札を捨てるとき
  SOUND_REVERSE = "Book1"       # カードを裏返すとき
  SOUND_CHANGE  = "Book2"       # 裏のカードを交換するとき
end
end


#/////////////////////////////////////////////////////////////////////////////#
#                                                                             #
#                下記のスクリプトを変更する必要はありません。                 #
#                                                                             #
#/////////////////////////////////////////////////////////////////////////////#


class << CAO::Poker
  #--------------------------------------------------------------------------
  # ● 所持コインの枚数を取得
  #--------------------------------------------------------------------------
  def coin
    return $game_variables[CAO::Poker::IDV_COIN]
  end
  #--------------------------------------------------------------------------
  # ● コインの増加（減少）
  #--------------------------------------------------------------------------
  def gain_coin(amount)
    $game_variables[CAO::Poker::IDV_COIN] += amount
    $game_variables[CAO::Poker::IDV_COIN] = 0 if self.coin < 0
  end
end

module Sound
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def self.define_poker_sound(method_name, file_name)
    if file_name.empty?
      instance_eval %Q{
        def #{method_name}
        end
      }
    else
      instance_eval %Q{
        @#{method_name} = RPG::SE.new(file_name, 80)
        def #{method_name}
          @#{method_name}.play
        end
      }
    end
  end
  #--------------------------------------------------------------------------
  # ● ポーカーの効果音
  #--------------------------------------------------------------------------
  define_poker_sound :play_poker_start,   CAO::Poker::SOUND_START
  define_poker_sound :play_poker_win,     CAO::Poker::SOUND_WIN
  define_poker_sound :play_poker_lose,    CAO::Poker::SOUND_LOSE
  define_poker_sound :play_poker_bet,     CAO::Poker::SOUND_BET
  define_poker_sound :play_poker_coin,    CAO::Poker::SOUND_COIN
  define_poker_sound :play_poker_deal,    CAO::Poker::SOUND_DEAL
  define_poker_sound :play_poker_slough,  CAO::Poker::SOUND_SLOUGH
  define_poker_sound :play_poker_reverse, CAO::Poker::SOUND_REVERSE
  define_poker_sound :play_poker_change,  CAO::Poker::SOUND_CHANGE
end

class CAO::Poker::Trump
  #--------------------------------------------------------------------------
  # ● スートの識別番号
  #--------------------------------------------------------------------------
  SUIT_SPADES   = 0
  SUIT_HEARTS   = 1
  SUIT_DIAMONDS = 2
  SUIT_CLUBS    = 3
  #--------------------------------------------------------------------------
  # ● 公開インスタンス変数
  #--------------------------------------------------------------------------
  attr_reader   :suit                     # スート (0..3)
  attr_reader   :number                   # 番号   (1..13)
  #--------------------------------------------------------------------------
  # ● オブジェクトの初期化
  #--------------------------------------------------------------------------
  def initialize(suit, number)
    @suit = suit
    @number = number
  end
  #--------------------------------------------------------------------------
  # ● トランプのＩＤ
  #--------------------------------------------------------------------------
  def id
    return 100 * @suit + @number
  end
  #--------------------------------------------------------------------------
  # ● デバッグ用
  #--------------------------------------------------------------------------
  def inspect
    return sprintf('%s%02d', '????'[@suit], @number)  # ????
  end
end

class CAO::Poker::Hand
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  @@deck = Array.new(13*4) {|i| CAO::Poker::Trump.new(i/13, i%13+1) }.freeze
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  attr_reader :hand                       # 手札
  attr_reader :stock                      # 山札
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def initialize
    @stock = @@deck.shuffle
    @hand = @stock.pop(5)
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def change(index)
    @hand[index] = @stock.pop
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def [](index)
    return @hand[index]
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def each
    @hand.each {|card| yield card }
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def sort!
    @hand.sort! {|a,b| a.number <=> b.number }
  end
  #--------------------------------------------------------------------------
  # ● 役の判定
  #--------------------------------------------------------------------------
  def final
    sort!       # 役を判定する前に手札のソートが必要
    royal    = _royal?
    straight = _straight?
    flush    = _flush?
    pairs    = _pairs
    pair_max = pairs.max
    pair_min = pairs.min
    pair_cnt = pairs.count(pair_max)
    
    return 9 if royal && flush                    # ロイヤルフラッシュ
    return 8 if straight && flush                 # ストレートフラッシュ
    return 7 if pair_max == 4                     # フォーカード
    return 6 if pair_max == 3 && pair_min == 2    # フルハウス
    return 5 if flush                             # フラッシュ
    return 4 if straight                          # ストレート
    return 3 if pair_max == 3 && pair_min == 1    # スリーカード
    return 2 if pair_max == 2 && pair_cnt == 4    # ツーペア
    return 1 if pair_max == 2                     # ワンペア
    return 0                                      # ハイカード
  end
  #--------------------------------------------------------------------------
  # ● ロイヤルストレートの判定
  #--------------------------------------------------------------------------
  def _royal?
    return @hand.map {|card| card.number } == [1, 10, 11, 12, 13]
  end
  #--------------------------------------------------------------------------
  # ● ストレートの判定
  #--------------------------------------------------------------------------
  def _straight?
    return true if _royal?
    @hand.each_with_index do |card,i|
      return false if card.number - @hand[0].number != i
    end
    return true
  end
  #--------------------------------------------------------------------------
  # ● フラッシュの判定
  #--------------------------------------------------------------------------
  def _flush?
    return @hand.all? {|card| card.suit == @hand[0].suit }
  end
  #--------------------------------------------------------------------------
  # ● ペア数を配列で取得 (重複する)
  #--------------------------------------------------------------------------
  def _pairs
    return @hand.map {|c1| @hand.count{|c2| c1.number == c2.number } }
  end
  protected :_royal?, :_straight?, :_flush?, :_pairs
end

class Game_Interpreter
  #--------------------------------------------------------------------------
  # ● ポーカー起動
  #--------------------------------------------------------------------------
  def start_poker
    SceneManager.call(Scene_Poker)
    Fiber.yield
  end
end

class Window_PokerHands < Window_Base
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  attr_reader :index
  #--------------------------------------------------------------------------
  # ● オブジェクト初期化
  #--------------------------------------------------------------------------
  def initialize
    super(0, 0, Graphics.width, fitting_height(5))
    clear
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def clear
    @index = -1
    @coin = 0
    refresh
  end
  #--------------------------------------------------------------------------
  # ● リフレッシュ
  #--------------------------------------------------------------------------
  def refresh
    self.contents.clear
    draw_all_items
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def draw_all_items
    10.times {|i| draw_item(i) }
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def draw_item(index)
    change_color(@index == index ? text_color(18) : normal_color)
    rect = Rect.new(0, 0, self.item_width, self.line_height)
    rect.x = (index / 5) * (rect.width + self.space)
    rect.y = (index % 5) * rect.height
    draw_text(rect, CAO::Poker::VOCAB_HANDS[index])
    draw_text(rect, prize_s(index), 2)
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def item_width
    return (contents_width - self.space) / 2
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def space
    return 16
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def prize(index = nil)
    return Integer(@coin * CAO::Poker::DIVIDEND_COVER[index || @index])
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def prize_s(index)
    if @coin == 0
      return "x#{CAO::Poker::DIVIDEND_COVER[index]}"
    else
      value = String(prize(index))
      value.sub!(/000$/, 'K') if CAO::Poker::ADDR_DIVIDEND && value.size > 4
      return value
    end
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def coin=(value)
    @coin = value
    refresh
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def index=(index)
    @index = index
    refresh
  end
end

class Window_Base
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def draw_poker_coin(x, y, width, value, text = "", enabled = true)
    change_color(system_color)
    draw_text(x, y, width, line_height, text) unless text.empty?
    draw_text(x, y, width, line_height, CAO::Poker::VOCAB_COIN_U, 2)
    change_color(normal_color, enabled)
    cx = text_size(CAO::Poker::VOCAB_COIN_U).width
    draw_text(x, y, width - cx - 2, line_height, value, 2)
  end
end

class Window_PokerCoin < Window_Base
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  WIDTH = Graphics.width / 8 * 3
  #--------------------------------------------------------------------------
  # ● オブジェクト初期化
  #--------------------------------------------------------------------------
  def initialize()
    super(0, 0, WIDTH, fitting_height(1))
    refresh
  end
  #--------------------------------------------------------------------------
  # ● リフレッシュ
  #--------------------------------------------------------------------------
  def refresh
    self.contents.clear
    draw_poker_coin(0,0,contents_width,CAO::Poker.coin,CAO::Poker::VOCAB_COIN)
  end
end

class Window_PokerBet < Window_Base
  #--------------------------------------------------------------------------
  # ● 公開インスタンス変数
  #--------------------------------------------------------------------------
  attr_reader   :number                   # 入力された数
  #--------------------------------------------------------------------------
  # ● オブジェクト初期化
  #--------------------------------------------------------------------------
  def initialize
    super(0, 0, 272, fitting_height(2))
    self.viewport = viewport
    @digits_max = CAO::Poker::MAX_BET % 10 + 1
    self.openness = 0
    deactivate
    reset
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def reset
    @count = 0
    @number = CAO::Poker::DEFAULT_COIN
    refresh
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def enable?
    @number <= CAO::Poker.coin
  end
  #--------------------------------------------------------------------------
  # ● リフレッシュ
  #--------------------------------------------------------------------------
  def refresh
    self.contents.clear
    change_color(normal_color)
    draw_text(0, 0, contents_width, line_height, CAO::Poker::VOCAB_BET, 1)
    draw_poker_coin(0, line_height, contents_width,
      sprintf("%0*d", @digits_max, @number), "", self.enable?)
  end
  #--------------------------------------------------------------------------
  # ● フレーム更新
  #--------------------------------------------------------------------------
  def update
    super
    process_digit_change
    process_handling
  end
  #--------------------------------------------------------------------------
  # ● 数字の変更処理
  #--------------------------------------------------------------------------
  def process_digit_change
    return unless open? && active
    if Input.repeat?(:UP) || Input.repeat?(:DOWN)
      last_number = @number
      if @count < 3
        @count += 1
        @number += CAO::Poker::BET_RATE if Input.repeat?(:UP)
        @number -= CAO::Poker::BET_RATE if Input.repeat?(:DOWN)
      else
        @number += CAO::Poker::BET_RATE * 10 if Input.repeat?(:UP)
        @number -= CAO::Poker::BET_RATE * 10 if Input.repeat?(:DOWN)
      end
      @number = CAO::Poker::BET_RATE if @number < CAO::Poker::BET_RATE
      @number = CAO::Poker::MAX_BET if CAO::Poker::MAX_BET < @number
      if @number != last_number
        Sound.play_poker_coin
        refresh
      end
    else
      @count = 0
    end
  end
  #--------------------------------------------------------------------------
  # ● 決定とキャンセルのハンドリング処理
  #--------------------------------------------------------------------------
  def process_handling
    return unless open? && active
    return process_ok     if Input.trigger?(:C)
    return process_cancel if Input.trigger?(:B)
  end
  #--------------------------------------------------------------------------
  # ● 決定ボタンが押されたときの処理
  #--------------------------------------------------------------------------
  def process_ok
    if self.enable?
      Sound.play_poker_bet
      deactivate
      close
      @ok_handler.call
    else
      Sound.play_buzzer
    end
  end
  #--------------------------------------------------------------------------
  # ● キャンセルボタンが押されたときの処理
  #--------------------------------------------------------------------------
  def process_cancel
    Sound.play_cancel
    @cancel_handler.call
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def set_ok_handler(method)
    @ok_handler = method
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def set_cancel_handler(method)
    @cancel_handler = method
  end
end

class Window_PokerMessage < Window_Base
  #--------------------------------------------------------------------------
  # ● オブジェクト初期化
  #--------------------------------------------------------------------------
  def initialize
    super(0, 0, 320, fitting_height(2))
    self.openness = 0
    deactivate
  end
  #--------------------------------------------------------------------------
  # ● フレーム更新
  #--------------------------------------------------------------------------
  def update
    super
    process_handling
  end
  #--------------------------------------------------------------------------
  # ● 決定とキャンセルのハンドリング処理
  #--------------------------------------------------------------------------
  def process_handling
    return unless active
    return process_ok     if Input.trigger?(:C)
    return process_cancel if Input.trigger?(:B)
  end
  #--------------------------------------------------------------------------
  # ● 決定ボタンが押されたときの処理
  #--------------------------------------------------------------------------
  def process_ok
    Sound.play_poker_slough
    deactivate
    close
    @handler.call
    Input.update
  end
  #--------------------------------------------------------------------------
  # ● キャンセルボタンが押されたときの処理
  #--------------------------------------------------------------------------
  def process_cancel
    process_ok
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def set_ok_handler(method)
    @handler = method
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def set_cancel_handler(method)
    @handler = method
  end
  #--------------------------------------------------------------------------
  # ● テキスト設定
  #--------------------------------------------------------------------------
  def set_text(text)
    if text != @text
      @text = text
      refresh
    end
  end
  #--------------------------------------------------------------------------
  # ● クリア
  #--------------------------------------------------------------------------
  def clear
    set_text("")
  end
  #--------------------------------------------------------------------------
  # ● リフレッシュ
  #--------------------------------------------------------------------------
  def refresh
    contents.clear
    draw_text_ex(4, 0, @text)
  end
end

class Window_PokerHelp < Window_Base
  #--------------------------------------------------------------------------
  # ● オブジェクト初期化
  #--------------------------------------------------------------------------
  def initialize
    super(0, 0, Graphics.width - Window_PokerCoin::WIDTH, fitting_height(1))
    refresh
  end
  #--------------------------------------------------------------------------
  # ● リフレッシュ
  #--------------------------------------------------------------------------
  def refresh
    self.contents.clear
    draw_text(0, 0, contents_width, contents_height, CAO::Poker::VOCAB_HELP)
  end
end

class Window_PokerManual < Window_Base
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def initialize
    bitmap = Bitmap.new(Graphics.width, Graphics.height)
    bitmap.fill_rect(bitmap.rect, Color.new(0, 0, 0, 160))
    @background = Sprite.new
    @background.bitmap = bitmap
    @background.visible = false
    super(0, 0, 424, fitting_height(CAO::Poker::MANUAL_LINE))
    self.openness = 0
    @background.z = self.z
    deactivate
    refresh
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def dispose
    @background.bitmap.dispose
    @background.dispose
    super
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def refresh
    self.contents.clear
    draw_text_ex(0, 0, CAO::Poker::TEXT_MANUAL)
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def update
    super
    @background.visible = !close?
    if open? && (Input.trigger?(:A) || Input.trigger?(:B))
      Sound.play_cancel
      @close_handler.call
    elsif close? && Input.trigger?(:A)
      Sound.play_ok
      @open_handler.call
    end
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def open_handler=(method)
    @open_handler = method
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def close_handler=(method)
    @close_handler = method
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def viewport=(value)
    @background.viewport = value
    super
  end
end

class Sprite_PokerTrump < Sprite
  #--------------------------------------------------------------------------
  # ● 定数
  #--------------------------------------------------------------------------
#  CARD_WIDTH  = Cache.system("PokerTrump").width  / 14
#  CARD_HEIGHT = Cache.system("PokerTrump").height / 4
  #--------------------------------------------------------------------------
  # ● 公開インスタンス変数
  #--------------------------------------------------------------------------
  attr_reader :wait_count
  #--------------------------------------------------------------------------
  # ● オブジェクト初期化
  #--------------------------------------------------------------------------
  def initialize(x, y, viewport)
    @pos_x = x
    @pos_y = y
    super(viewport)
    self.x = x
    self.y = y
    self.opacity = 0
    create_backface
    centering_origin
    init_instance_variables
    hide
  end
  #--------------------------------------------------------------------------
  # ● オブジェクト解放
  #--------------------------------------------------------------------------
  def dispose
    @trump_bitmap.dispose if @trump_bitmap
    super
  end
  #--------------------------------------------------------------------------
  # ● カードの裏柄
  #--------------------------------------------------------------------------
  def create_backface
    @backface = Bitmap.new(CARD_WIDTH, CARD_HEIGHT)
    @backface.blt(0, 0, Cache.system("PokerTrump"),
      Rect.new(CARD_WIDTH * 13, CARD_HEIGHT * 0, CARD_WIDTH, CARD_HEIGHT))
    self.bitmap = @backface
  end
  #--------------------------------------------------------------------------
  # ● インスタンス変数の初期化
  #--------------------------------------------------------------------------
  def init_instance_variables
    @trump = nil
    @opening = @closing = false
    @openness = 100
    @wait_count = 0
    @target_bitmap = nil
    @target_x = self.x
    @target_y = self.y
    @target_opacity = self.opacity
    @speed_x = 0.0
    @speed_y = 0.0
    @speed_opacity = 0.0
  end
  #--------------------------------------------------------------------------
  # ● 転送元原点を中央にする
  #--------------------------------------------------------------------------
  def centering_origin
    self.ox = CARD_WIDTH / 2
    self.oy = CARD_HEIGHT / 2
  end
  #--------------------------------------------------------------------------
  # ● フレーム更新
  #--------------------------------------------------------------------------
  def update
    super
    if @wait_count > 0
      @wait_count -= 1
    else
      # update_deal_sound
      update_reverse
      update_position
    end
  end
  #--------------------------------------------------------------------------
  # ● カード表示時に効果音を鳴らす (未使用)
  #--------------------------------------------------------------------------
  def update_deal_sound
    if @start_deal
      Sound.play_poker_deal
      @start_deal = false
    end
  end
  #--------------------------------------------------------------------------
  # ● カードの向きを更新
  #--------------------------------------------------------------------------
  def update_reverse
    return if self.x != @target_x
    return if self.y != @target_y
    return if self.opacity != @target_opacity
    if @closing
      @openness -= 20
      if @openness <= 0
        self.bitmap = @target_bitmap
        @openness = 0
        @closing = false
      end
      self.zoom_x = @openness / 100.0
    elsif @openning
      @openness += 20
      if @openness >= 100
        @openness = 100
        @openning = false
      end
      self.zoom_x = @openness / 100.0
    end
  end
  #--------------------------------------------------------------------------
  # ● カードの位置を更新
  #--------------------------------------------------------------------------
  def update_position
    return if @openness != 0 && @openness != 100
    if self.x != @target_x
      @x += @speed_x
      self.x = @x
      if @speed_x < 0
        self.x = @target_x if self.x < @target_x
      else
        self.x = @target_x if self.x > @target_x
      end
    end
    if self.y != @target_y
      @y += @speed_y
      self.y = @y
      if @speed_y < 0
        self.y = @target_y if self.y < @target_y
      else
        self.y = @target_y if self.y > @target_y
      end
    end
    if self.opacity != @target_opacity
      @opacity += @speed_opacity
      self.opacity = @opacity
    end
  end
  #--------------------------------------------------------------------------
  # ● カードを表向きにする
  #--------------------------------------------------------------------------
  def open(wait = -1)
    self.wait = wait unless wait < 0
    @target_bitmap = @trump_bitmap
    reverse_motion
  end
  #--------------------------------------------------------------------------
  # ● カードを裏向きにする
  #--------------------------------------------------------------------------
  def close
    @target_bitmap = @backface
    reverse_motion
  end
  #--------------------------------------------------------------------------
  # ● カードを反転する
  #--------------------------------------------------------------------------
  def reverse
    front? ? close : open
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def reverse_motion
    @openness = 100
    @closing = true
    @openning = true
  end
  #--------------------------------------------------------------------------
  # ● カードの表示予約
  #--------------------------------------------------------------------------
  def deal
    @start_deal = true    # 未使用 (update_deal_sound)
    self.bitmap = @backface
    self.x = @pos_x - 4
    self.y = @pos_y - 4
    @target_x = @pos_x
    @target_y = @pos_y
    @target_opacity = 255
    calc_speed(10)
  end
  #--------------------------------------------------------------------------
  # ● カードの消去予約
  #--------------------------------------------------------------------------
  def slough
    @target_x = self.x
    @target_y = self.y - 20
    @target_opacity = 0
    calc_speed(10)
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def calc_speed(frame)
    @x = self.x
    @y = self.y
    @opacity = self.opacity
    @speed_x = (@target_x - self.x) / frame.to_f
    @speed_y = (@target_y - self.y) / frame.to_f
    @speed_opacity = (@target_opacity - self.opacity) / frame.to_f
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def show
    self.opacity = 255
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def hide
    self.opacity = 0
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def in_motion?
    return true if reversing?
    return true if moving?
    return false
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def reversing?
    return @closing || @openning
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def moving?
    return true if self.x != @target_x
    return true if self.y != @target_y
    return true if self.opacity != @target_opacity
    return false
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def wait?
    return @wait_count != 0
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def front?
    return self.bitmap != @backface
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def trump=(trump)
    return if @trump == trump
    @trump = trump
    @trump_bitmap.dispose if @trump_bitmap
    @trump_bitmap = Bitmap.new(CARD_WIDTH, CARD_HEIGHT)
    rect = Rect.new(0, 0, CARD_WIDTH, CARD_HEIGHT)
    rect.x = CARD_WIDTH * (trump.number - 1)
    rect.y = CARD_HEIGHT * trump.suit
    @trump_bitmap.blt(0, 0, Cache.system("PokerTrump"), rect)
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def wait=(value)
    @wait_count = value
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def select
    self.oy = CARD_HEIGHT / 2 + 8
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def unselect
    self.oy = CARD_HEIGHT / 2
  end
  #--------------------------------------------------------------------------
  # ● 原点は固定、拡大率はカードをめくる動作で使用
  #--------------------------------------------------------------------------
  protected :ox=, :oy=, :zoom_x=, :zoom_y=
end

class Spriteset_PokerTrump
  #--------------------------------------------------------------------------
  # ● 定数
  #--------------------------------------------------------------------------
  CARD_MAX = 5
  WAIT_TIME = 8
  #--------------------------------------------------------------------------
  # ● 公開インスタンス変数
  #--------------------------------------------------------------------------
  attr_reader   :index                    # カーソル位置
  attr_reader   :active                   # 
  #--------------------------------------------------------------------------
  # ● オブジェクト初期化
  #--------------------------------------------------------------------------
  def initialize(viewport)
    @wait_count = -1
    @viewport = viewport
    @viewport.z = 50
    cw = @viewport.rect.width / CARD_MAX
    sx = @viewport.rect.width / 2 - cw * 2
    sy = @viewport.rect.height / 2
    @card_sprites = Array.new(CARD_MAX) do |i|
      Sprite_PokerTrump.new(sx + cw * i, sy, @viewport)
    end
    clear_handler
    unselect
    deactivate
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def dispose
    @viewport.dispose
    @card_sprites.each {|sp| sp.dispose }
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def update
    @viewport.update
    @card_sprites.each {|sp| sp.update }
    update_timer
    update_cursor
    if self.active
      process_ok                           if Input.trigger?(:C)
      process_cancel                       if Input.trigger?(:B)
      cursor_right(Input.trigger?(:RIGHT)) if Input.repeat?(:RIGHT)
      cursor_left (Input.trigger?(:LEFT))  if Input.repeat?(:LEFT)
    end
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def deal(hand)
    Sound.play_poker_deal
    CARD_MAX.times do |i|
      @card_sprites[i].trump = hand[i]
      @card_sprites[i].wait = WAIT_TIME * i
      @card_sprites[i].deal
    end
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def slough
    @card_sprites.each {|sp| sp.slough }
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def open(hand = nil)
    if hand
      CARD_MAX.times {|i| @card_sprites[i].trump = hand[i] }
      list = @card_sprites.select {|sp| !sp.front? }
    else
      list = @card_sprites
    end
    list.each_with_index {|sp,i| sp.open(WAIT_TIME * i) }
    return self
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def reversed_indexes
    result = []
    @card_sprites.each_with_index do |sp,i|
      result << i unless sp.front?
    end
    return result
  end
  #--------------------------------------------------------------------------
  # ● ウィンドウのアクティブ化
  #--------------------------------------------------------------------------
  def activate
    @active = true
    self
  end
  #--------------------------------------------------------------------------
  # ● ウィンドウの非アクティブ化
  #--------------------------------------------------------------------------
  def deactivate
    @active = false
    self
  end
  #--------------------------------------------------------------------------
  # ● 項目の選択
  #--------------------------------------------------------------------------
  def select(index)
    @index = index
    update_cursor
  end
  #--------------------------------------------------------------------------
  # ● 項目の選択解除
  #--------------------------------------------------------------------------
  def unselect
    @index = -1
    update_cursor
  end
  #--------------------------------------------------------------------------
  # ● ハンドラの設定をクリア
  #--------------------------------------------------------------------------
  def clear_handler
    @handler = {}
  end
  #--------------------------------------------------------------------------
  # ● 動作に対応するハンドラの設定
  #     method : ハンドラとして設定するメソッド (Method オブジェクト)
  #--------------------------------------------------------------------------
  def set_handler(symbol, method)
    @handler[symbol] = method
  end
  #--------------------------------------------------------------------------
  # ● ハンドラの存在確認
  #--------------------------------------------------------------------------
  def handle?(symbol)
    @handler.include?(symbol)
  end
  #--------------------------------------------------------------------------
  # ● ハンドラの呼び出し
  #--------------------------------------------------------------------------
  def call_handler(symbol)
    @handler[symbol].call if handle?(symbol)
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def update_cursor
    @card_sprites.each {|sp| sp.unselect }
    @card_sprites[@index].select unless @index < 0
  end
  #--------------------------------------------------------------------------
  # ● カーソルの移動可能判定
  #--------------------------------------------------------------------------
  def cursor_movable?
    return self.active
  end
  #--------------------------------------------------------------------------
  # ● カーソルを右に移動
  #--------------------------------------------------------------------------
  def cursor_right(wrap = false)
    if index < CARD_MAX - 1 || wrap
      Sound.play_cursor
      select((index + 1) % CARD_MAX)
    end
  end
  #--------------------------------------------------------------------------
  # ● カーソルを左に移動
  #--------------------------------------------------------------------------
  def cursor_left(wrap = false)
    if index > 0 || wrap
      Sound.play_cursor
      select((index - 1 + CARD_MAX) % CARD_MAX)
    end
  end
  #--------------------------------------------------------------------------
  # ● 選択項目のカードを取得
  #--------------------------------------------------------------------------
  def current_card
    return @index < 0 ? nil : @card_sprites[@index]
  end
  #--------------------------------------------------------------------------
  # ● 決定ボタンが押されたときの処理
  #--------------------------------------------------------------------------
  def process_ok
    Sound.play_poker_reverse
    Input.update
    unless @index < 0
      @card_sprites[@index].reverse
    end
  end
  #--------------------------------------------------------------------------
  # ● キャンセルボタンが押されたときの処理
  #--------------------------------------------------------------------------
  def process_cancel
    Sound.play_poker_change
    Input.update
    deactivate
    call_handler(:cancel)
  end
  #--------------------------------------------------------------------------
  # ● すべてのカードの動作が完了するまでの時間
  #--------------------------------------------------------------------------
  def wait_count
    return @card_sprites.max_by {|sp| sp.wait_count }.wait_count
  end
  #--------------------------------------------------------------------------
  # ● ビューポートのサイズ
  #--------------------------------------------------------------------------
  def viewport
    return @viewport.rect
  end
  #--------------------------------------------------------------------------
  # ● 移動中判定
  #--------------------------------------------------------------------------
  def update_timer
    return if @wait_count < 0
    return if @card_sprites.any? {|card| card.in_motion? }
    @wait_count -= 1
    @deferred.call if @wait_count < 0 && @deferred
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def wait(duration, method = nil)
    @wait_count = duration
    @deferred = method
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def waiting?
    return @wait_count >= 0
  end
end

class Scene_Poker < Scene_MenuBase
  #--------------------------------------------------------------------------
  # ● 開始処理
  #--------------------------------------------------------------------------
  def start
    super
    create_hands_window
    create_help_window
    create_gold_window
    create_card_sprite
    create_info_windows
    start_game
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def create_hands_window
    @hands_window = Window_PokerHands.new
    @hands_window.x = (Graphics.width - @hands_window.width) / 2
    @hands_window.y = 32
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def create_help_window
    @help_window = Window_PokerHelp.new
    @help_window.y = Graphics.height - @help_window.height
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def create_gold_window
    @coin_window = Window_PokerCoin.new
    @coin_window.x = Graphics.width - @coin_window.width
    @coin_window.y = Graphics.height - @coin_window.height
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def create_card_sprite
    viewport = Viewport.new
    viewport.rect.y = @hands_window.y + @hands_window.height
    viewport.rect.height -= viewport.rect.y + @help_window.height
    @spriteset = Spriteset_PokerTrump.new(viewport)
    @spriteset.set_handler(:cancel, method(:change_card))
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def create_info_windows
    @info_viewport = Viewport.new
    @info_viewport.z = 100
    create_bet_window
    create_message_window
    create_manual_window
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def create_bet_window
    @bet_window = Window_PokerBet.new
    @bet_window.viewport = @info_viewport
    @bet_window.x = (@info_viewport.rect.width - @bet_window.width) / 2
    @bet_window.y = (@info_viewport.rect.height - @bet_window.height) / 2
    @bet_window.y = @spriteset.viewport.y + 16
    @bet_window.set_ok_handler(method(:on_bet_ok))
    @bet_window.set_cancel_handler(method(:return_scene))
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def create_message_window
    @message_window = Window_PokerMessage.new
    @message_window.viewport = @info_viewport
    @message_window.x = (@info_viewport.rect.width - @message_window.width) / 2
    @message_window.y = (@info_viewport.rect.height-@message_window.height) / 2
    @message_window.y = @spriteset.viewport.y + 16
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def create_manual_window
    @manual_window = Window_PokerManual.new
    @manual_window.viewport = @info_viewport
    @manual_window.x = (@info_viewport.rect.width - @manual_window.width) / 2
    @manual_window.y = (@info_viewport.rect.height - @manual_window.height) / 2
    @manual_window.open_handler = proc do
      @manual_window.open
    end
    @manual_window.close_handler = proc do
      @manual_window.close
    end
  end
  #--------------------------------------------------------------------------
  # ● 終了処理
  #--------------------------------------------------------------------------
  def terminate
    super
    @info_viewport.dispose
    @spriteset.dispose
  end
  #--------------------------------------------------------------------------
  # ● フレーム更新
  #--------------------------------------------------------------------------
  def update
    super
    @spriteset.update if @manual_window.close?
  end
  #--------------------------------------------------------------------------
  # ● 全ウィンドウの更新
  #--------------------------------------------------------------------------
  def update_all_windows
    if @manual_window.close?
      super
    else
      @manual_window.update
    end
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def start_game
    @hands_window.clear
    @hand = CAO::Poker::Hand.new
    @spriteset.deal(@hand)
    @spriteset.wait(0, method(:start_bet))
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def end_game
    @spriteset.slough
    @spriteset.wait(10, method(:start_game))
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def start_bet
    @bet_window.refresh
    @bet_window.open.activate
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def start_select
    Sound.play_poker_start
    @spriteset.select(0)
    @spriteset.activate
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def show_mes_win
    Sound.play_poker_win
    CAO::Poker.gain_coin(@hands_window.prize)
    @coin_window.refresh
    @message_window.set_text(CAO::Poker::VOCAB_WIN % @hands_window.prize)
    @message_window.open.activate
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def show_mes_lose
    Sound.play_poker_lose
    @message_window.set_text(CAO::Poker::VOCAB_LOSE % @hands_window.prize)
    @message_window.open.activate
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def on_bet_ok
    CAO::Poker.gain_coin(-@bet_window.number)
    @coin_window.refresh
    @hands_window.coin = @bet_window.number
    @bet_window.close
    @spriteset.open
    @spriteset.wait(0, method(:start_select))
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def on_mes_change_ok
    start_select
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def on_mes_win_ok
    end_game
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def on_mes_lose_ok
    end_game
  end
  #--------------------------------------------------------------------------
  # ● 
  #--------------------------------------------------------------------------
  def change_card
    @spriteset.unselect
    @spriteset.reversed_indexes.each {|index| @hand.change(index) }
    @spriteset.open(@hand)
    @hands_window.index = @hand.final
    if @bet_window.number <= @hands_window.prize
      @message_window.set_ok_handler(method(:on_mes_win_ok))
      @message_window.set_cancel_handler(method(:on_mes_win_ok))
      @spriteset.wait(0, method(:show_mes_win))
    else
      @message_window.set_ok_handler(method(:on_mes_lose_ok))
      @message_window.set_cancel_handler(method(:on_mes_lose_ok))
      @spriteset.wait(0, method(:show_mes_lose))
    end
  end
end
